/* $Id: e1562tput.h,v 1.8 1998/07/20 22:49:20 ericb Exp $ */
/* Copyright (C) 1995 - 1998, Hewlett-Packard Company, all rights reserved. */
/* Written by Bryan Murray */

#ifndef __e1562tput_INCLUDED__
#define __e1562tput_INCLUDED__ 1

/* This file (e1562tput.h) and the corresponding C file e1562tput.c provide
 * a set of functions which can be used to perform throughput records to
 * a LIF filesystem on an E1562.  The SICL library and the E1562 LIF library
 * must already be installed on the host system in order to use these
 * routines.  All filenames referenced in the following routines must
 * comply with filename and volume specifications outlined in e1562lif.h.
 *
 * No special LIF utilities are provided in this function set, as it is
 * expected that users of these routines will have the E1562 LIF library
 * and the associated utilities.  These utilities will provide a means
 * of creating a LIF file system, reading the directory, copying files
 * to and from the file system, etc.
 */

#include <e1562lif.h>   /* need e1562_FILE */


#ifdef __cplusplus
extern "C" {
#endif

#if defined(MSDOS) || defined(_MSDOS) || defined(_WINDOWS)
#ifndef MSDOS
#define MSDOS
#endif
#define TPUTPROG __export __pascal
#else
#define TPUTPROG
#endif




/* Define the default id to use when accessing the E1562 LIF library.
 */
#define E1562_ID 3


/* Define a data type used to interface with the E1562 specific routines
 * to open LIF files and perform throughput record and playbacks.
 */
typedef short E1562Handle;



/* Open access to the E1562.  This function must be called before any
 * interaction with the E1562.  This function does a SICL iopen(),
 * resets the E1562 and initializes the LIF library.  The parameter
 * interface is usually "vxi", but will depend upon how the host system
 * is configured.  The returned parameter, handle, is passed to every
 * other e1562tput_ routine.  It is a private structure which contains
 * information which needs to be preserved across function calls.
 */
extern long TPUTPROG e1562tput_open
    (const char* interface, short e1562LogicalAddress, E1562Handle* handle);

/* Terminate access to the E1562.  The LIF library is closed and the E1562
 * SICL id is issued an iclose().  Use of the E1562 handle after the call
 * to this routine is not allowed and may cause unexpected failures.
 */
extern long TPUTPROG e1562tput_close
    (E1562Handle e1562Handle);

/* Open an E1562 LIF file in preparation for a throughput record.  The
 * total file size must be known at open time, as a LIF file cannot be
 * extended in size after it has been created.  This means that the value
 * of totalBytes must include the size of the data to be recorded, the
 * size of any header information, and the size of any information
 * following the recorded data.  The value passed to totalBytes will
 * be padded with one SCSI blocksize, since the recorded data must start
 * at a SCSI block boundary.  This will mean that any directory listing
 * may show a different size for the file than what was passed to this
 * function.  Since this routine may be called with just a filename allowing
 * the volume to be defaulted, the full filename is returned to the calling
 * function by modifying the filename buffer.  It is possible to default the
 * volume name only if a single disk is used as the LIF volume, or if a pair
 * of disks, one on each SCSI bus of the E1562, is used as the LIF volume.
 * Also, the SCSI devices must be the lowest addressed devices on the SCSI
 * bus.  If the volume name is specified, any number of SCSI devices may be
 * used to make up the file system.  This function expects that a file system
 * already exists on the devices -- this can be done using the e1562in command
 * from the E1562 LIF utilites.  Since the filename may be defaulted, the full
 * filename is returned in case the file needs to be opened again later.
 */
extern long TPUTPROG e1562tput_openRecordFile
    (E1562Handle e1562Handle, char* filename, double totalBytes);

/* Open an existing file for modification.
 */
extern long e1562tput_openFileForUpdate
    (E1562Handle e1562Handle,
     char* fullFileName);

/* Initialize the E1562 sequence to perform a local bus throughput with
 * the specified number of inputs and blocksize.  A constant blocksize for
 * all channels is assumed.  An E1562 session is setup to write at the
 * next SCSI block boundary in the file opened in e1562tput_openRecordFile().
 * The open file is then closed, as the session is used for the throughput
 * instead of the file.  Any header information to be written to the file
 * should be written before calling this function as the current position
 * of the file pointer is used to determine where to start writing the data.
 * The number of bytes needed for padding is returned so that the offset in
 * the file to the recorded data can be determined.  If the pad is non-zero,
 * the header should be written again at the end of the record.
 */
extern long TPUTPROG e1562tput_setupRecord
    (E1562Handle e1562Handle, unsigned long bytesPerInputBlock,
     short numberInputs, unsigned long* retPadBytes);

/* Start the E1562 record sequence running.  If a callback function is
 * specified, an SRQ is setup such that a function call will be made
 * to the specified routine when the throughput record is finished.  The
 * specified argument and the number of bytes recorded will be passed
 * to the callback routine.  If no callback is desired, callbackFunc may
 * be specified as 0.
 */
extern long TPUTPROG e1562tput_startRecord
    (E1562Handle e1562Handle, double lengthInBytes,
     void (*callbackFunc)(E1562Handle, volatile void*, double),
     volatile void* callbackArg);

extern long TPUTPROG e1562tput_recordBytes
    (E1562Handle e1562Handle, double* bytes);

extern long TPUTPROG e1562tput_recordFinished
    (E1562Handle e1562Handle, long* flag);

/* Open an E1562 LIF file in preparation for a throughput post-process.
 */
extern long TPUTPROG e1562tput_openPlaybackFile
    (E1562Handle e1562Handle, char* fullFileName);

/* Initialize the E1562 sequence to perform a playback using an E1562
 * sequence to read bytes into the shared RAM on the E1562.  Synchronization
 * between the host program and the E1562 is done through the use of the
 * first 2 bytes of E1562 shared RAM as a flag.  By using the routine
 * e1562tput_readPlaybackScan(), all synchronization is encapsulated in
 * these example routines.  Be aware that the maximum value for bytesPerScan
 * is 262142, although there are no checks for exceeding this value.  A
 * session is setup to be used with the E1562 sequence.  The open playback
 * file is closed.  The function e1562tput_readPlaybackScan is used to
 * perform the required synchronization with the E1562 after
 * e1562tput_startPlayback has been called.
 */
extern long TPUTPROG e1562tput_setupPlayback
    (E1562Handle e1562Handle, unsigned long dataOffset,
     unsigned long bytesPerScan);

/* Start the E1562 playback sequence running.  If a callback function is
 * specified, an SRQ is setup such that a function call will be made
 * to the specified routine when the throughput playback is finished.  The
 * specified argument and the number of bytes read will be passed
 * to the callback routine.  The argument firstScanOffset is the scan
 * number of the first scan when the throughput file was treated as a
 * circular buffer -- it is zero otherwise.  If a callback function is
 * not needed, pass 0 as the argument callbackFunc.
 */
extern long TPUTPROG e1562tput_startPlayback
    (E1562Handle e1562Handle, double lengthInBytes,
     unsigned long firstScanOffset,
     void (*callbackFunc)(E1562Handle, volatile void*, double),
     volatile void* callbackArg);

/* Perform the required synchronization with the E1562 to get bytes
 * transferred into its shared RAM and copy the bytes into the host.  The
 * value byteCount should be less than or equal to 262142 and should be the
 * same as the value bytesPerScan passed to e1562tput_setupPlayback.
 */
extern long TPUTPROG e1562tput_readPlaybackScan
    (E1562Handle e1562Handle, unsigned long byteCount, void* buf);

/* Seek to an absolute location in a file.
 */
extern long TPUTPROG e1562tput_seek
    (E1562Handle e1562Handle, double dataOffset);

extern long TPUTPROG e1562tput_write
    (void* buf, size_t elsize, size_t count, E1562Handle e1562Handle);

extern long TPUTPROG e1562tput_read
    (void* buf, size_t elsize, size_t count, E1562Handle e1562Handle);

/* Allow the application to put the E1562 local bus into reset, or take
 * it out of reset.  This is needed to provide for the safe reset of all
 * devices on the local bus.  For example, a safe reset consists of first
 * placing all adjacent local bus devices into reset, then from left to
 * right in the VXI cardcage, take each device's local bus out of reset.
 */
extern long TPUTPROG e1562tput_resetE1562lbus
    (E1562Handle e1562Handle, short putIntoReset);

/* Halt an E1562 sequence.  This can be used to abort a throughput record
 * before it would normally complete, or in the case of a circular buffer,
 * to stop the throughput record.  This function may also be used to stop
 * a playback sequence early and place the E1562 in an idle state.  For both
 * a record or playback sequence, if a callback function was specified, it
 * is called with the number of bytes which were actually throughout.
 */
extern long TPUTPROG e1562tput_abortThroughput
    (E1562Handle e1562Handle);

/* Provide for the ability to implement a circular buffer in a throughput
 * file.  For instance, with the use of a callback routine when the throughput
 * is finished, this function will allow the sequence to be restarted at the
 * beginning of the recorded data.  Some application logic is required to
 * determine when to stop restarting the sequence, or when to abort it.
 */
extern long TPUTPROG e1562tput_continueThroughput
    (E1562Handle e1562Handle);

#ifdef __cplusplus
}
#endif

#endif /* __e1562tput_INCLUDED__ */
